/* $Id: modgroup.c,v 1.132 1999/03/10 22:26:59 ericb Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Taken from E1431 library, heavily modified by Eric Backus */

/* This files contains all of the routines to handle creating,
   deleting and other manipulations of channel groups.  A channel
   group is one or more E1432 channels, either from the same or from
   distinct modules, grouped together under a groupID such that they
   can be setup easily with only one function call for the whole
   group.  Most functions in the E1432 library have as their first
   parameter a groupID to reference a particular grouping of modules.
   Most of these functions alternatively accept a chanID, in order to
   reference a single channel.  When a function accepts both forms,
   there is only one parameter called ID, and this is unambiguous
   since groupIDs are sequential negative integers, starting with -1,
   and chanIDs are sequential positive integers, starting with +1.

   Note that the channel numbers seen by the user start at one (for
   each type of input channel).  However, the channel index passed to
   a Semaphore module starts at zero. */

/* global data storage allocated in this module */
#define NOEXTERN_FLAG


#include "sema.h"

/*
 *********************************************************************
 Reset the module.  This function doesn't wait for the module to
 recover, since that might take a long time.  Instead, call
 i1432_wait_reset() to do the wait.
 *********************************************************************
 */
/*ARGSUSED*/
SHORTSIZ16
i1432_reset(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 error;

    error = i1432_direct_write_register(mn, E1432_CONTROL_REG,
					(SHORTSIZ16)
					(E1432_CONTROL_A24_ENABLE |
					 E1432_CONTROL_SYSFAIL_INHIBIT_ON));
    if (error)
	return error;
    error = i1432_direct_write_register(mn, E1432_CONTROL_REG,
					(SHORTSIZ16)
					(E1432_CONTROL_A24_ENABLE |
					 E1432_CONTROL_SYSFAIL_INHIBIT_ON |
					 E1432_CONTROL_RESET_ON));
    if (error)
	return error;
    error = i1432_direct_write_register(mn, E1432_CONTROL_REG,
					(SHORTSIZ16)
					(E1432_CONTROL_A24_ENABLE |
					 E1432_CONTROL_SYSFAIL_INHIBIT_ON));
    if (error)
	return error;

    return 0;
}

/*
 *********************************************************************
 Wait for module to come out of reset.  We would like to undo SYSFAIL
 INHIBIT in the VXI Control register, which is probably set as we
 enter this routine.  But we can't do that, because the PASSED bit may
 not yet be set in the VXI Status register if we are booting from host
 port.  Instead, it is the responsibility of the install function or
 program to clear the SYSFAIL INHIBIT bit.
 *********************************************************************
 */
/*ARGSUSED*/
SHORTSIZ16
i1432_wait_reset(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 error, dummy;

    /* If the B bus was locked before the reset, the processor doesn't
       wake up until we do a B-bus peek. */
    error = i1432_direct_read_register(mn, E1432_IRQ_STATUS2_REG,
				       &dummy);
    if (error)
	return error;

    /* Let bootrom come alive, if that's what we're booting from. */
    i1432_pause(0.005);

    /* Reset is done, but the module will now need firmware via
       e1432_install before it can do anything. */
    return 0;
}

#ifndef HAVE_VTL /* all I/O is in vtlio.c  for VTP plug&Play */

/* This talks only to A16 space, and just sets up pointers so
   that i1432_sicl_setup2 can talk to A24 space. */
SHORTSIZ16
i1432_sicl_setup1(E1432_MODULE_LIST_NODE *mn, int big_map, int extra_io)
{
    SHORTSIZ16 error;
#ifdef	HPVXI_DOWNLOAD
    unsigned int map_pages;
#endif

#ifdef	HAVE_SICL
    mn->a16_base = (SHORTSIZ16 *) (void *)
	((char *) i1432_a16_base + (64 * mn->la) + 0xc000);
#endif
#ifdef	HPVXI_DOWNLOAD
    mn->a16_base = (SHORTSIZ16 *)(void *)os_card_address((LONGSIZ32)(mn->la));
    if ( mn->a16_base == 0 )
    {
        DIAG_PRINTF(1, ("  Failure to map module A16 space.\n"))
        DIAG_PRINTF(1, ("  Check:  Card plugged in.\n"));
        DIAG_PRINTF(1, ("          Logical Address switches S3 should be"
          " set to %d.\n", mn->la));
        DIAG_PRINTF(1, ("  Suspect:  VXI interface U780, its ROM U707.\n"));
	return ERR1432_UNABLE_TO_MAP_MEMORY;
    }
#endif

    /* See if the module is hung */
    if ( extra_io == E1432_EXTRA_IO_NONE )
    {
	/* We don't want to abort an e1432_install() if the 96002 has
	   the B bus locked, since the e1432_install() will do a reset
	   which should fix the problem.  But we do want to abort the
	   install if there's nothing there at all. */
        error = i1432_direct_ping_register(mn, E1432_ID_REG);
        if (error) return error;
    }
    else if ( extra_io == E1432_EXTRA_IO )
    {
	/* This register is a more stringent test, since it requires
	   the VXI interface to actually do a bus cycle on the 96002 B
	   bus.  If the 96002 is hung on that bus, this will bus
	   error. */
        error = i1432_direct_ping_register(mn, E1432_IRQ_STATUS2_REG);
        if (error) return error;
    }
    else  /* E1432_EXTRA_IO_DIAG */
    {
	/* more diagnostic approach */
        USHORTSIZ16 data;

	/* The "ping" seems to be necessary to catch there being no card
	   at the LA.  Could be the pipeline latency of the bus error on
	   the series 700 workstations. */
        error = i1432_direct_ping_register(mn, E1432_ID_REG);
	if ( ! error )
	{
            error = i1432_direct_read_register(mn, E1432_ID_REG,
	      (SHORTSIZ16 *)&data);
	}
	if ( error )
	{
	    DIAG_PRINTF(1, ("  An attempt to read the E1432 ID register"
	      " failed"));
	    if ( error == ERR1432_BUS_ERROR )
	    {
	        DIAG_PRINTF(1, (" in a bus error.\n"))
	    }
	    else
	    {
	        DIAG_PRINTF(1, (" in error %d.\n", error));
	    }
	    DIAG_PRINTF(1, ("  This was the first read of the E1432.\n"));
	    DIAG_PRINTF(1, ("\n"));
	    DIAG_PRINTF(1, ("  Check:  Card plugged in.\n"));
	    DIAG_PRINTF(1, ("          Logical Address switches S3 should be"
	      " set to %d.\n", mn->la));
	    DIAG_PRINTF(1, ("  Suspect:  VXI interface U780, its ROM U707.\n"));
	    return error;
        }
        /* check ID register response */
        if (data != 0xcfff)
	{
	    DIAG_PRINTF(1, ("  The E1432 ID register returned the wrong"
	      " value.\n"));
	    DIAG_PRINTF(1, ("  This was the first read of the E1432.\n"));
	    DIAG_PRINTF(1, ("  Expected 0xcfff, got 0x%4.4x\n", data));
	    DIAG_PRINTF(1, ("\n"));
	    DIAG_PRINTF(1, ("  Suspect:  VXI interface U780, its ROM U707.\n"));
	    return i1432_la_print_error(mn->la, ERR1432_NO_MOD_AT_LA);
	}
    }

    if (!i1432_sema_present(mn))
	return i1432_la_print_error(mn->la, ERR1432_NO_MOD_AT_LA);

    mn->a24_base = NULL;
    mn->a24_fifo_base = NULL;
#ifdef	NO_D32
    mn->d32 = 0;
#else
    mn->d32 = 1;
#endif

#ifdef	HAVE_SICL
    error = i1432_sicl_imap_setup(mn, big_map);
    if (error)
	return error;
#endif
#ifdef	HPVXI_DOWNLOAD
    if (big_map && !mn->a24_256k)
	map_pages = 16;
    else
	map_pages = 4;
    mn->pos_a24_handle = os_setup_a24(mn->la, 0, map_pages);
    if ( mn->pos_a24_handle == NULL )
    {
	return i1432_la_print_error(mn->la, ERR1432_UNABLE_TO_MAP_MEMORY);
    }
#endif

    return 0;
}

/* This part of the setup actually talks to hardware, so it might try
   to unmap A24 memory from other modules, so only do it after all
   modules have properly initialized their A24 pointers in
   i1432_sicl_setup1. */
SHORTSIZ16
i1432_sicl_setup2(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 error;

    /* This will get done inside i1432_direct_write_register, so this
       is not really needed here.  But the error reporting is better
       this way, and this might slightly speed things up later if the
       mapping is already done. */
    error = i1432_imap(mn, 0);
    if (error)
	return error;

    /* Write to Page Map register, to point to B Bus */
    error = i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);
    if (error)
	return error;

    /* Enable local bus */
    error = i1432_direct_write_register(mn, E1432_PORT_CTL_REG,
					E1432_LBUS_ENABLE);
    if (error)
	return error;

    return 0;
}

SHORTSIZ16
i1432_sicl_cleanup(E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 error;

    error = i1432_iunmap(mn, 0);
    if (error)
	return error;

#ifdef	HPVXI_DOWNLOAD
    /* os_unsetup_a24(mn->pos_a24_handle);  unresolved ref - unsupported? */
    mn->pos_a24_handle = NULL;
#endif

    return 0;
}

SHORTSIZ16
i1432_fake_setup_sicl(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 la,
		      struct i1432_chan_info *chan_list_ptr,
		      int extra_io)
{
    SHORTSIZ16 error;

    /* Clean up previous stuff, if needed */
    error = i1432_sicl_cleanup(mn);
    if (error)
	return error;

    /* Fake out some library functions into thinking we already did an
       e1432_assign_channel_numbers */
    mn->la = la;
    chan_list_ptr->mn = mn;
    i1432_chan_list[0] = chan_list_ptr;
    i1432_chan_count[0] = 1;

    /* Big map so e1432_zap can work, no fifo because we don't need it */
    error = i1432_sicl_setup1(mn, 1, extra_io);
    if (error)
	return error;
    error = i1432_sicl_setup2(mn);
    if (error)
	return error;

    return 0;
}

SHORTSIZ16
i1432_fake_cleanup_sicl(E1432_MODULE_LIST_NODE *mn,
			struct i1432_chan_info *chan_list_ptr,
			int chan_count)
{
    SHORTSIZ16 error;

    i1432_chan_list[0] = chan_list_ptr;
    i1432_chan_count[0] = chan_count;

    error = i1432_sicl_cleanup(mn);
    if (error)
	return error;

    return 0;
}

#endif /* !HAVE_VTL */


SHORTSIZ16 EXPORT
e1432_get_hwconfig(SHORTSIZ16 modCount, SHORTSIZ16 * las,
		   struct e1432_hwconfig *list)
{
    struct i1432_chan_info tmp_chan_list;
    struct i1432_chan_info *save_chan_list;
    E1432_MODULE_LIST_NODE node;
    E1432_MODULE_LIST_NODE *mn = &node;
    SHORTSIZ16 error, dtype, trig, error2;
    LONGSIZ32 fw_rev, hw_rev, bootrom_rev, oct, dram, lbus;
    LONGSIZ32 id, sca_rev, bob_id, type, nchan, tmp;
    int     i, j, save_chan_count;

    TRACE_PRINTF(0, ("e1432_get_hwconfig(%d, 0x%p, 0x%p)\n",
		     modCount, las, list));

    /* check for bad modCount parameter */
    if (modCount <= 0 || modCount >= E1432_MODULES)
	return i1432_print_error(ERR1432_ILLEGAL_MODULE_COUNT);

    error = 0;

    /* Do this before calling i1432_fake_setup_sicl */
#if defined(HAVE_SICL) || defined(HAVE_VTL)
    mn->sicl_id = 0;
#endif

    mn->a24_base = NULL;
    save_chan_list = i1432_chan_list[0];
    save_chan_count = i1432_chan_count[0];

    /* Check that these really are Semaphores, map in A24 space */
    for (i = 0; i < modCount; i++, list++)
    {
	list->man_id = 0;
	list->model_code = 0;

	error = i1432_fake_setup_sicl(mn, las[i], &tmp_chan_list, 1);
	if (error)
	    goto cleanup;

	/* See if firmware is responding */
	/* Pass in bogus hw and channel ID, which works because of
	   kludges done in i1432_fake_setup_sicl */
	error = e1432_get_auto_trigger(0, 1, &trig);
	if (error)
	{
	    goto cleanup;
	}
	/* Don't just call e1432_set_auto_trigger, because it won't do
	   anything since the value we're setting is the value that
	   already is there.  Use i1432_write_cmd1 to force the
	   handshake with the firmware. */
	error = i1432_write_cmd1(0, 1, E1432_CMD_SET_AUTO_TRIGGER,
				 (LONGSIZ32) trig);
	/* Ignore most errors, which are probably just leftover errors
	   from previous measurements.  Most of the time,
	   e1432_get_hwconfig is used to determine if we need to
	   re-install firmware, and we don't in most cases.  However,
	   the following errors indicate that something major is
	   wrong, so don't ignore them. */

	if (error == ERR1432_NO_GROUP ||
	    error == ERR1432_NO_CHANNEL ||
	    error == ERR1432_SICL_ERROR ||
	    error == ERR1432_BUS_ERROR ||
	    error == ERR1432_STATUS_READ_TIMEOUT)
	    goto cleanup;

	/* Get hardware revision */
	error = i1432_direct_read32_register(mn, E1432_HW_REV_REG,
					     &hw_rev);
	if (error)
	    goto cleanup;

	/* Get bootrom revision */
	error = i1432_direct_read32_register(mn,
					     E1432_BOOTROM_REV_REG,
					     &bootrom_rev);
	if (error)
	    goto cleanup;

	/* Get firmware revision */
	error = i1432_direct_read32_register(mn, E1432_FW_REV_REG,
					     &fw_rev);
	if (error)
	    goto cleanup;

	/* Get model code */
	error = i1432_direct_read_register(mn, E1432_DEVICE_TYPE_REG,
					   &dtype);
	if (error)
	    goto cleanup;

	/* Get LBUS present */
	error = i1432_direct_read32_register(mn,
					     E1432_OCTAVE_PRESENT_REG,
					     &oct);
	if (error)
	    goto cleanup;

	/* Get DRAM size */
	error = i1432_direct_read32_register(mn, E1432_DRAM_SIZE_REG,
					     &dram);
	if (error)
	    goto cleanup;

	/* Get LBUS present */
	error = i1432_direct_read32_register(mn,
					     E1432_LBUS_PRESENT_REG,
					     &lbus);
	if (error)
	    goto cleanup;

	/* Get serial string */
	for (j = 0; j < E1432_SERIAL_LENGTH; j += 4)
	{
	    error = i1432_direct_read32_register(mn,
						 E1432_SERIAL_START + j,
						 &tmp);
	    if (error)
		goto cleanup;

	    /* E1432 stores the serial string big-endian */
	    list->serial[j    ] = (char) ( tmp >> 24        );
	    list->serial[j + 1] = (char) ((tmp >> 16) & 0xff);
	    list->serial[j + 2] = (char) ((tmp >>  8) & 0xff);
	    list->serial[j + 3] = (char) ( tmp        & 0xff);
	}
	/* Make sure it's terminated */
	list->serial[E1432_SERIAL_LENGTH - 1] = '\0';

	list->man_id = E1432_MAN_ID;
	list->model_code = dtype & DTYPE_MODEL_CODE_MASK;
	list->hw_rev = hw_rev;
	list->bootrom_rev = bootrom_rev;
	list->fw_rev = fw_rev;
	list->total_chans = 0;
	list->input_chans = 0;
	list->source_chans = 0;
	list->tach_chans = 0;
	list->oct_present = (int)oct;
	list->lbus_present = (int)lbus;
	list->dram_size = dram;
	list->a24_used = mn->a24_256k ? 262144L : 1048576L;

	for (j = 0; j < E1432_SCAS; j++)
	{
	    /* Is there an SCA here? */
	    error = i1432_direct_read32_register
		(mn, E1432_SINFO_ID_REG + j * E1432_SINFO_REG_OFFSET,
		 &id);
	    if (error)
		goto cleanup;

	    /* Get the rev, type and number of chans */
	    error = i1432_direct_read32_register
		(mn, E1432_SINFO_REV_REG + j *
		 E1432_SINFO_REG_OFFSET, &sca_rev);
	    if (error)
		id = E1432_SCA_ID_NONE;
	    error = i1432_direct_read32_register
		(mn, E1432_SINFO_BOB_ID_REG + j *
		 E1432_SINFO_REG_OFFSET, &bob_id);
	    if (error)
		id = E1432_SCA_ID_NONE;
	    error = i1432_direct_read32_register
		(mn, E1432_SINFO_TYPE_REG + j *
		 E1432_SINFO_REG_OFFSET, &type);
	    if (error)
		id = E1432_SCA_ID_NONE;
	    error = i1432_direct_read32_register
		(mn, E1432_SINFO_CHANS_REG + j *
		 E1432_SINFO_REG_OFFSET, &nchan);
	    if (error)
		id = E1432_SCA_ID_NONE;

	    list->sca_id[j] = (int) id;
	    if (id == E1432_SCA_ID_NONE)
	    {
		list->sca_rev[j] = 0;
		list->bob_id[j] = 0;
		continue;
	    }

	    list->sca_rev[j] = sca_rev;
	    list->bob_id[j] = (int) bob_id;
	    switch (type)
	    {
	    case E1432_CHAN_TYPE_INPUT:
		list->input_chans += (int)nchan;
		break;
	    case E1432_CHAN_TYPE_SOURCE:
		list->source_chans += (int)nchan;
		break;
	    case E1432_CHAN_TYPE_TACH:
		list->tach_chans += (int)nchan;
		break;
	    case E1432_CHAN_TYPE_OTHER:
		break;
	    }
	    list->total_chans += (int)nchan;
	}
    }

 cleanup:
    error2 = i1432_fake_cleanup_sicl(mn, save_chan_list, save_chan_count);
    if (error2)
	error = error2;
    return error;
}

static SHORTSIZ16
preset_help(E1432ID hw, SHORTSIZ16 ID)
{
#ifndef HPVXI_DOWNLOAD  /* these not in HPVXI_DOWNLOAD at the moment */
    SHORTSIZ16 error;

    error = e1432_set_auto_group_meas(hw, ID,
				      E1432_AUTO_GROUP_MEAS_DEF);
    if (error)
	return error;

    error = e1432_set_auto_range_mode(hw, ID,
				      E1432_AUTO_RANGE_MODE_DEF);
    if (error)
	return error;

    error = e1432_set_interrupt_mask(hw, ID, 0);
    if (error)
	return error;
    error = e1432_set_interrupt_priority(hw, ID,
					 E1432_IRQ_PRIORITY_NONE);
    if (error)
	return error;
#endif /* HPVXI_DOWNLOAD */

    return 0;
}

/*
 *********************************************************************
 This function checks to see if a E1432 module is at each of the
 logical addresses in <logAddrArray>, and assigns channel numbers to
 all channels found in the E1432 modules.

 If it fails to find a E1432 module at the logical address, or can't
 allocate memory for the new structure, it will return a negative
 error number, otherwise it returns 0.

 Channel IDs and module IDs are allocated following the order of
 logical addresses provided - i.e. first la gets modID 1, and chanIDs
 1 through 8, second la gets modID 2, and chanIDs 9 through 16, etc.
 This call should be made once, since successive calls to it will
 delete the previously existing channel and modules data structures.

 If the <nmod> parameter is zero, it means to free the existing
 internal structures and then exit.

 If the <reset> parameter is non-zero, it means to preset the modules
 before returning.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_assign_channels(SHORTSIZ16 nmod, SHORTSIZ16 *la_list,
		      E1432ID *hw, int preset)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;
    LONGSIZ32 nchan, id, type;
    int     i, j, chan, index;

    TRACE_PRINTF(0, ("e1432_assign_channels(%d, 0x%p, 0x%p, %d)\n",
		     nmod, la_list, hw, preset));


    /* Check for bad nmod parameter */
    if (nmod < 0 || nmod >= E1432_MODULES)
	return i1432_print_error(ERR1432_ILLEGAL_MODULE_COUNT);


    /* Free old module list if needed */
    /* NOTE: i1432_mod_list is initialized to NULL in init_io_driver() */
    if (i1432_mod_list != NULL)
    {
	mn = i1432_mod_list;
	for (i = 0; i < i1432_mod_count; i++, mn++)
	{
	    error = i1432_unmap_one_fifo(mn);
	    if (error)
		return error;
#ifdef HAVE_VTL
	    error = i1432_visa_cleanup(mn);
#else
	    error = i1432_sicl_cleanup(mn);
#endif
	    if (error)

		return error;
	}
	free(i1432_mod_list);
	i1432_mod_list = NULL;
	i1432_mod_count = 0;
    }

    /* Free old channel list if needed and init channel counts */
    /* NOTE: i1432_chan_list is initialized to NULL in init_io_driver() */
    for (i = 0; i < E1432_CHAN_TYPES; i++)
    {
	if (i1432_chan_list[i] != NULL)
	{
	    free(i1432_chan_list[i]);
	    i1432_chan_list[i] = NULL;
	}
	i1432_chan_count[i] = 0;
    }

    /* nmod of zero means all we do is free the formerly-used
       memory, and then exit. */
    if (nmod == 0)
	return 0;

    /* Allocate space for list of modules */
    i1432_mod_list = malloc(nmod * sizeof *i1432_mod_list);
    if (i1432_mod_list == NULL)
	return i1432_print_error(ERR1432_MALLOC);
    i1432_mod_count = nmod;

    *hw = (E1432ID) 0x87654321;

    /* Check that these really are Semaphores, map in A24 space */
    mn = i1432_mod_list;
    for (i = 0; i < nmod; i++, mn++)
    {
	/* First half of setup */
	/* This talks only to A16 space, and just sets up pointers so
	   that i1432_sicl_setup2 can talk to A24 space. */
	mn->hw = *hw;
	mn->la = la_list[i];
#ifndef HAVE_VTL
	error = i1432_sicl_setup1(mn, 0, 1);
	if (error)
	    return error;
#endif
    }
    mn = i1432_mod_list;
    for (i = 0; i < nmod; i++, mn++)
    {
	/* Second half of setup */
	/* The i1432_sicl_setup2 actually talks to hardware, so it
	   might try to unmap A24 memory from other modules, so only
	   do it after all modules have properly initialized their A24
	   pointers in i1432_sicl_setup1. */
#ifdef HAVE_VTL
	error = i1432_visa_setup(mn);
#else
	error = i1432_sicl_setup2(mn);
#endif
	if (error)
	    return error;

	/* Initialize some of the fields of the module node */
#if defined(INSTR_CALLBACKS) && defined(HAVE_VTL)
	mn->cbFunc = NULL;
	mn->cbValue = 0;
	mn->cbHandle = 0;
#endif
	mn->rs_bsize = 0;
	mn->rs_navg = 0;
	mn->rs_span = 0;
	mn->rs_magsq = 0;
	mn->rs_ready = 0;
	mn->rs_req = 0;
	mn->rs_irq_done = 0;
	mn->rs_irq_error = 0;
	mn->rs_irq_id = 0;
	mn->rs_irq_off = 0;
	mn->rs_irq_save_line = -1;
	mn->rs_irq_save_mask = -1;
#ifdef	HAVE_SICL
	mn->rs_irq_sicl_id = 0;
#endif
	mn->autoGroup = 0;
	mn->blocksize = 0;
	mn->append_status = 0;
	mn->data_size = 0;
	mn->meas_type = 0;
	mn->freq_mag_squared = 0;
	mn->zoom = 0;
	mn->order_tracking = 0;
	mn->rpm_arm = 0;

	/* The E1431 library did a hard reset here, but that's not
	   good for E1432 because we would then have to download new
	   firmware. Instead we do a global preset below if the
	   "preset" parameter is non-zero. */
    }

    /* Count the number of chans, so we can malloc chan lists */
    mn = i1432_mod_list;
    for (i = 0; i < nmod; i++, mn++)
    {
	mn->nchan = 0;
	mn->input_chans = 0;
	for (j = 0; j < E1432_SCAS; j++)
	{
	    /* Is there an SCA here? */
	    error = i1432_direct_read32_register
		(mn, E1432_SINFO_ID_REG + j * E1432_SINFO_REG_OFFSET,
		 &id);
	    if (error)
            {
		return error;
            }
	    if (id == E1432_SCA_ID_NONE)
		continue;

	    /* Get the type and number of chans */
	    error = i1432_direct_read32_register
		(mn, E1432_SINFO_TYPE_REG + j *
		 E1432_SINFO_REG_OFFSET, &type);
	    if (error)
            {
		return error;
            }
	    error = i1432_direct_read32_register
		(mn, E1432_SINFO_CHANS_REG + j *
		 E1432_SINFO_REG_OFFSET, &nchan);
	    if (error)
            {
		return error;
            }

	    if (type < 0 || type > E1432_CHAN_TYPES ||
		nchan < 0 || nchan > E1432_CHANS)
            {
		return i1432_la_print_error(mn->la, ERR1432_IO);
            }

	    i1432_chan_count[type] += (int)nchan;
	    mn->nchan += (int)nchan;
	    if(type == E1432_CHAN_TYPE_INPUT)
		mn->input_chans += (int)nchan;
	}
    }

    /* Malloc the channel info lists */
    for (i = 0; i < E1432_CHAN_TYPES; i++)
	if (i1432_chan_count[i] > 0)
	{
	    i1432_chan_list[i] =
		malloc(i1432_chan_count[i] *
		       sizeof *(i1432_chan_list[i]));
	    if (i1432_chan_list[i] == NULL)
		return i1432_print_error(ERR1432_MALLOC);
	}

    /* Go through module list again, recording info about channels */
    for (i = 0; i < E1432_CHAN_TYPES; i++)
	i1432_chan_count[i] = 0;
    mn = i1432_mod_list;
    for (i = 0; i < nmod; i++, mn++)
    {
	/* Tell module the first input channel number, used in trailer */
	error = i1432_direct_write32_register
	    (mn, E1432_INPUT_CHAN_START_REG,
	     i1432_chan_count[E1432_CHAN_TYPE_INPUT]);
	if (error)
	    return error;

	mn->chan_id = 0;
	index = 0;
	for (j = 0; j < E1432_SCAS; j++)
	{
	    /* Is there an SCA here? */
	    error = i1432_direct_read32_register
		(mn, E1432_SINFO_ID_REG + j * E1432_SINFO_REG_OFFSET,
		 &id);
	    if (error)
		return error;
	    if (id == E1432_SCA_ID_NONE)
		continue;

	    /* Get the type, id, and number of chans */
	    error = i1432_direct_read32_register
		(mn, E1432_SINFO_TYPE_REG + j *
		 E1432_SINFO_REG_OFFSET, &type);
	    if (error)
		return error;
	    error = i1432_direct_read32_register
		(mn, E1432_SINFO_CHANS_REG + j *
		 E1432_SINFO_REG_OFFSET, &nchan);
	    if (error)
		return error;

	    if (index == 0)
		mn->chan_id = (SHORTSIZ16)
		    (i1432_chan_count[type] + 1 +
		     (type << E1432_CHAN_TYPE_SHIFT));
	    for (chan = i1432_chan_count[type];
		 chan < i1432_chan_count[type] + nchan;
		 chan++, index++)
	    {
		i1432_chan_list[type][chan].mn = mn;
		i1432_chan_list[type][chan].index = index;
		i1432_chan_list[type][chan].sca = j;
		i1432_chan_list[type][chan].sca_id = (int) id;
		i1432_chan_list[type][chan].auto_range_mode =
		    E1432_AUTO_RANGE_MODE_DEF;
		i1432_chan_list[type][chan].active = 0;
		i1432_chan_list[type][chan].calc_data = 0;
		i1432_chan_list[type][chan].scale = 0;

		if (nchan < 2)
		    /* No other channel */
		    i1432_chan_list[type][chan].other_chan = 0;
		else if (chan == i1432_chan_count[type])
		    /* First in SCA, use next channel for other channel */
		    i1432_chan_list[type][chan].other_chan =
			(SHORTSIZ16)
			(chan + 2 + (type << E1432_CHAN_TYPE_SHIFT));
		else
		    /* Use previous channel for other channel */
		    i1432_chan_list[type][chan].other_chan =
			(SHORTSIZ16)
			(chan + (type << E1432_CHAN_TYPE_SHIFT));
	    }
	    i1432_chan_count[type] += (int) nchan;
	}
    }

    if (preset)
    {
	/* Send a global preset.  Precede it with reset_measure_start
	   to ensure that multi-module systems don't run into
	   problems. */
	mn = i1432_mod_list;
	for (i = 0; i < nmod; i++, mn++)
	    if (mn->nchan > 0)
	    {
		chan = i1432_get_chan_from_module(mn);
		error = i1432_write_cmd0(*hw, (SHORTSIZ16)chan,
					 E1432_CMD_RESET_MEASURE_START);
		if (error)
		    return error;
	    }
	mn = i1432_mod_list;
	for (i = 0; i < nmod; i++, mn++)
	    if (mn->nchan > 0)
	    {
		chan = i1432_get_chan_from_module(mn);
		error = i1432_write_cmd0(*hw, (SHORTSIZ16)chan,
					 E1432_CMD_GLOBAL_PRESET);
		if (error)
		    return error;

		error = preset_help(*hw, (SHORTSIZ16) chan);
		if (error)
		    return error;
	    }

	/* Remove any existing channel groups */
	/* Doesn't seem like this should be necessary, the user should
	   be responsible for this, but that's what the E1431 library
	   does. */
	error = e1432_delete_all_chan_groups(*hw);
	if (error)
	    return error;
    }

    return 0;
}

SHORTSIZ16 EXPORT
e1432_assign_channel_numbers(SHORTSIZ16 nmod, SHORTSIZ16 *la_list,
			     E1432ID *hw)
{
    TRACE_PRINTF(0, ("e1432_assign_channel_numbers(%d, 0x%p, 0x%p)\n",
		     nmod, la_list, hw));

    return e1432_assign_channels(nmod, la_list, hw, 1);
}

/*
 *********************************************************************
 Set the flag for all module nodes referenced by ID.
 *********************************************************************
 */
static void
i1432_flag_modules(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    SHORTSIZ16 error;
    int     mod;

    /* clear all module flags */
    mn = i1432_mod_list;
    for (mod = 0; mod < i1432_mod_count; mod++, mn++)
	mn->flag = 0;

    /* for each channel in group find module and set flag */
    if (ID < 0)
    {
	gn = i1432_group_list;
	/* traverse channel list */
	while (gn)
	{
	    if (gn->groupID == ID)
	    {
		cn = gn->chanlist;
		while (cn)
		{
		    error = i1432_get_module_from_chan(hw, cn->chanID,
						       &mn);
		    if (error)
			return;
		    mn->flag = 1;
		    cn = cn->next;
		}
	    }
	    gn = gn->next;
	}
    }
    /* for a channel ID find module and set flag */
    else
    {
	error = i1432_get_module_from_chan(hw, ID, &mn);
	if (error)
	    return;
	mn->flag = 1;
    }
}

static SHORTSIZ16
update_modlist(E1432ID hw, E1432_GROUP_LIST_NODE *gn)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;
    int     i, nmod;

    /* Free old module list */
    free(gn->modlist);

    /* Count how many modules are in the group */
    i1432_flag_modules(hw, gn->groupID);
    mn = i1432_mod_list;
    nmod = 0;
    for (i = 0; i < i1432_mod_count; i++, mn++)
	if (mn->flag)
	    nmod++;
    gn->modcount = nmod;

    /* Allocate an array for the modules in the group */
    gn->modlist = (E1432_MODULE_LIST_NODE **)
	malloc(gn->modcount * sizeof (E1432_MODULE_LIST_NODE *));
    if (gn->modlist == NULL && nmod > 0)
	return i1432_print_error(ERR1432_MALLOC);

    /* Initialize the module array */
    mn = i1432_mod_list;
    nmod = 0;
    for (i = 0; i < i1432_mod_count; i++, mn++)
	if (mn->flag)
	    gn->modlist[nmod++] = mn;

    /* System module needs to be reset, since we might have deleted
       the system module. */
    error = i1432_assign_system_module(hw, gn->groupID);
    if (error)
	return error;

    /* There might be a new module that needs to be told there is no
       trigger master.  But that causes problems if there is some
       other group that has a trigger master, and that group contains
       channels that are also in this group.  To be safe, we can't do
       anything here if gn->trigmn is NULL.  There could be problems
       if somebody removes a module from a group with a trigger
       master, and then adds that module to a group with no trigger
       master.  What can be done about this? */

/* do not have e1432_set_trigger_master() in SCPI download */
#ifndef HPVXI_DOWNLOAD
    if (gn->trigmn != NULL)
    {
	/* Trigger master needs to be reset, since we might have added
	   a new module or deleted the master module. */
	for (i = 0; i < nmod; i++)
	    if (gn->trigmn == gn->modlist[i])
		break;
	/* If i == nmod, then the trigger master no longer in
	   group, and we pick another trigger master arbitrarily.
	   If the trigger master is still in the group, there
	   might be a new module that needs to be told it's a
	   trigger slave. */
	error = e1432_set_trigger_master(hw, (SHORTSIZ16)(i == nmod ?
					 gn->groupID :
					 gn->trigmn->chan_id),
					 E1432_TRIGGER_MASTER_ON);
	if (error)
	    return error;
    }
#endif /* HPVXI_DOWNLOAD */

    return 0;
}

/*
 *********************************************************************
 Add a channel to a channel group.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_channel_group_add(E1432ID hw, SHORTSIZ16 group_id,
			SHORTSIZ16 chan_id)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn, *new_cn, *prev_cn;
    int     error, type, count, new_type, new_count;

    /* Check the IDs */
    gn = i1432_get_group_node(hw, group_id);
    if (!gn)
	return i1432_print_error(ERR1432_NO_GROUP);
    error = i1432_checkID(hw, chan_id);
    if (error)
	return error;
    if (chan_id < 0)
	return i1432_print_error(ERR1432_NO_CHANNEL);

    /* See if channel is already in the list */
    cn = gn->chanlist;
    while (cn)
    {
	if (cn->chanID == chan_id)
	    return 0;	/* Found it, so nothing to do */
	cn = cn->next;
    }

    /* Create new channel node */
    new_cn = (E1432_CHAN_LIST_NODE *)
	malloc(sizeof (E1432_CHAN_LIST_NODE));
    if (new_cn == NULL)
	return i1432_print_error(ERR1432_MALLOC);
    new_cn->chanID = chan_id;

    new_type = chan_id & E1432_CHAN_TYPE_MASK;
    new_count = chan_id & E1432_CHAN_MASK;

    /* Add channel to list */
    prev_cn = NULL;
    cn = gn->chanlist;
    while (cn)
    {
	type = cn->chanID & E1432_CHAN_TYPE_MASK;
	count = cn->chanID & E1432_CHAN_MASK;
	if (type == new_type && count > new_count)
	    /* New channel goes after prev, before current channel */
	    break;
	prev_cn = cn;
	cn = cn->next;
    }

    new_cn->next = cn;
    if (prev_cn == NULL)
	/* New channel is at start of list */
	gn->chanlist = new_cn;
    else
	/* New channel is not at start of list */
	prev_cn->next = new_cn;

    return update_modlist(hw, gn);
}

/*
 *********************************************************************
 Delete a channel from a channel group.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_channel_group_remove(E1432ID hw, SHORTSIZ16 group_id,
			   SHORTSIZ16 chan_id)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn, *prev_cn;
    int     error;

    /* Check the IDs */
    gn = i1432_get_group_node(hw, group_id);
    if (!gn)
	return i1432_print_error(ERR1432_NO_GROUP);
    error = i1432_checkID(hw, chan_id);
    if (error)
	return error;
    if (chan_id < 0)
	return i1432_print_error(ERR1432_NO_CHANNEL);

    /* Scan channel list */
    prev_cn = NULL;
    cn = gn->chanlist;
    while (cn)
    {
	if (cn->chanID == chan_id)
	    break;
	prev_cn = cn;
	cn = cn->next;
    }
    if (cn == NULL)
	return 0;		/* Channel already not in list */

    /* Remove from list */
    if (prev_cn == NULL)
	/* Channel was at start of list */
	gn->chanlist = cn->next;
    else
	/* Channel was not at start of list */
	prev_cn->next = cn->next;
    free(cn);

    return update_modlist(hw, gn);
}

/*
 *********************************************************************
 Creates a new channel group.  It returns POSITIVE error number if an
 error occurs otherwise it returns a groupID (negative number) that is
 used for all subsequent references to this new group.  This
 convention seems bogus, but it IS convenient that group IDs and
 channels IDs are of opposite sign.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_create_channel_group(E1432ID hw, SHORTSIZ16 nchan,
			   SHORTSIZ16 * list)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    SHORTSIZ16 prev[E1432_CHAN_TYPES];
    SHORTSIZ16 error;
    int     i, type, lastid;

    TRACE_PRINTF(0, ("e1432_create_channel_group(0x%p, %d, 0x%p)\n",
		     hw, nchan, list));

	
    /* Verify the channel IDs */
    if (nchan < 0)
	return -i1432_print_error(ERR1432_NO_CHANNEL);
    for (i = 0; i < nchan; i++)
    {
	if (list[i] < 0)
	    return -i1432_print_error(ERR1432_NO_CHANNEL);
	error = i1432_checkID(hw, list[i]);
	if (error)
	    return -error;
    }

    /* Check for ascending channel order within each channel type */
    for (i = 0; i < E1432_CHAN_TYPES; i++)
	prev[i] = 0;
    for (i = 0; i < nchan; i++)
    {
	type = (list[i] & E1432_CHAN_TYPE_MASK) >>
	    E1432_CHAN_TYPE_SHIFT;
	if ((prev[type] & E1432_CHAN_MASK) >=
	    (list[i] & E1432_CHAN_MASK))
	    return -i1432_print_error(ERR1432_ILLEGAL_CHANNEL_ORDER);
	prev[type] = list[i];
    }

    /* make a changroup node */
    gn = (E1432_GROUP_LIST_NODE *)
	malloc(sizeof (E1432_GROUP_LIST_NODE));
    if (gn == NULL)
	return -i1432_print_error(ERR1432_MALLOC);
    /* get last group id */
    if (i1432_group_list == 0)
	lastid = 0;
    else
	lastid = i1432_group_list->groupID;
    /* place it at head of changrouplist */
    gn->next = i1432_group_list;
    i1432_group_list = gn;
    gn->chanlist = NULL;
    /* modcount initialized in update_modlist */
    gn->modlist = NULL;
    gn->groupID = lastid - 1;	/* assign next group id */
    /* system_mod initialized in update_modlist */
    gn->trigmn = NULL;
    gn->otra = 0;

    /* make a channel node for each channel in list */
    /* and string it onto chanlist pointer */
    for (i = nchan - 1; i >= 0; i--)
    {
	cn = (E1432_CHAN_LIST_NODE *)
	    malloc(sizeof (E1432_CHAN_LIST_NODE));
	if (cn == NULL)
	    return -i1432_print_error(ERR1432_MALLOC);
	cn->chanID = list[i];
	/* place channel node at front of list */
	cn->next = gn->chanlist;
	gn->chanlist = cn;
    }

    error = update_modlist(hw, gn);
    if (error)
	return -error;

    /* The E1431 library did a reset here.  This seems undesirable, so
       I'm not going to do it.  But we do need to set auto group
       meas, and all channels should be set to active. */
    error = e1432_set_auto_group_meas(hw, gn->groupID,
				      E1432_AUTO_GROUP_MEAS_DEF);
    if (error)
	return -error;

    error = e1432_set_active(hw, gn->groupID, E1432_ACTIVE_DEF);
    if (error)
	return -error;

    return gn->groupID;
}

/*
 *********************************************************************
 This function removes a channel group with the id, <groupID>.  The
 entry for this id is removed from the chanGroupID array and the
 dynamic memory associated with the module group is freed.  It returns
 -1 if there is no module group associated with groupID, otherwise it
 returns 0.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_delete_channel_group(E1432ID hw, SHORTSIZ16 ID)
{
    SHORTSIZ16 error;
    E1432_GROUP_LIST_NODE *gn, *lastgn, *nextgn;
    E1432_CHAN_LIST_NODE *cn, *nextcn;

    TRACE_PRINTF(0, ("e1432_delete_channel_group(0x%p, %d)\n",
		     hw, ID));

    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    gn = i1432_get_group_node(hw, ID);
    if (!gn)
	return i1432_print_error(ERR1432_NO_GROUP);

    /* Free the array of modules in the group */
    free(gn->modlist);

    /* remove all chanlist nodes strung onto group node */
    cn = gn->chanlist;
    while (cn)
    {
	/* free this channel node */
	nextcn = cn->next;
	free(cn);
	cn = nextcn;
    }
    /* remove groupnode at head of list */
    if (i1432_group_list == gn)
    {
	i1432_group_list = gn->next;
	free(gn);
    }
    else
    {
	/* remove group node in list */
	lastgn = i1432_group_list;
	nextgn = i1432_group_list->next;
	/* walk down link list until nextgn = gn */
	while (nextgn != gn)
	{
	    nextgn = nextgn->next;
	    lastgn = lastgn->next;
	}
	/* now connect next pointer to skip over gn */
	lastgn->next = nextgn->next;
	free(gn);
    }
    return 0;
}

/*
 *********************************************************************
 This function frees all memory associated all channel groups.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_delete_all_chan_groups(E1432ID hw)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_GROUP_LIST_NODE *temp;
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_delete_all_chan_groups(0x%p)\n", hw));
    gn = i1432_group_list;
    while (gn)
    {
	temp = gn;
	gn = gn->next;
	error = e1432_delete_channel_group(hw, temp->groupID);
	if (error)
	    return error;
    }
    return 0;
}

/* Because it is just like all other parameter get functions,
   e1432_get_active is now found in parms.c */

SHORTSIZ16 EXPORT
e1432_set_active(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 state)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;
    LONGSIZ32 tmp;
    int     cindex;

    TRACE_PRINTF(0, ("e1432_set_active(0x%p, %d, %d)\n",
		     hw, ID, state));

    /* check for valid id */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    /* check if channel or group */
    if (ID < 0)
    {
	/* iterate thru group */
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;
	while (cn)
	{
	    error = e1432_set_active(hw, cn->chanID, state);
	    if (error)
		return error;
	    cn = cn->next;
	}
    }
    else
    {
	/* set lone channel */
	cindex = i1432_get_chan_index(hw, ID);

	/* Getting a parameter value is fast compared to setting one,
	   so first check to see if the new value will be identical to
	   the old one.  If so, then don't bother setting the
	   parameter. */
	error = e1432_read32_register(hw, ID, E1432_ACTIVE_CREG +
				      cindex * E1432_CREG_OFFSET,
				      &tmp);
	if (error)
	    return error;
	if (tmp == state)
	    return 0;

	error = i1432_write_cmd2(hw, ID, E1432_CCMD_SET_ACTIVE,
				 cindex, state);
	if (error)
	    return error;

	/* clear settling flag when changing */
	error = i1432_get_module_from_chan(hw, ID, &mn);
	if (error)
	    return error;
	mn->wasACsettled[cindex] = 0;
    }
    return 0;
}

/*
 *********************************************************************
 Reset modules.  Returns null if error.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_reset(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error;
    int     mod;

    TRACE_PRINTF(0, ("e1432_reset(0x%p, %d)\n", hw, ID));

    /* Check ID */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    /* Reset all modules with this ID */
    if (ID < 0)
    {
	/* Iterate through group */
	gn = i1432_get_group_node(hw, ID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    /* Do one module */
	    mn = gn->modlist[mod];
	    error = i1432_reset(mn);
	    if (error)
		return error;
	}

	/* Wait for all modules to recover from reset */
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    /* Do one module */
	    mn = gn->modlist[mod];
	    error = i1432_wait_reset(mn);
	    if (error)
		return error;
	}
    }
    else
    {
	error = i1432_get_module_from_chan(hw, ID, &mn);
	if (error)
	    return error;
	error = i1432_reset(mn);
	if (error)
	    return error;
	error = i1432_wait_reset(mn);
	if (error)
	    return error;
    }

    return 0;
}

SHORTSIZ16 EXPORT
e1432_preset(E1432ID hw, SHORTSIZ16 ID)
{
    SHORTSIZ16 error;

    TRACE_PRINTF(0, ("e1432_preset(0x%p, %d)\n", hw, ID));

    error = preset_help(hw, ID);
    if (error)
	return error;

    /* Note unused parameter */
    return i1432_set_chan(hw, ID, -1, E1432_CCMD_PRESET, 0);
}

/*
 *********************************************************************
 Returns a pointer to the e1432_group_list_node given its groupID.
 Returns null if parameter is not a group ID.
 *********************************************************************
 */
/*ARGSUSED*/
E1432_GROUP_LIST_NODE *
i1432_get_group_node(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_GROUP_LIST_NODE *gn;

    if (ID >= 0)
	return NULL;
    gn = i1432_group_list;
    /* loop until end of link list or found ID */
    while ((gn) && (gn->groupID != ID))
	gn = gn->next;
    return gn;
}

/*
 *********************************************************************
 Return a pointer to the modulenode given a channel number.  Full
 error checking is done.
 *********************************************************************
 */
/*ARGSUSED*/
SHORTSIZ16
i1432_get_module_from_chan(E1432ID hw, SHORTSIZ16 ID,
			   E1432_MODULE_LIST_NODE **mnp)
{
    if ( ID < 0 )
    {
	/* Previously errored if passed a group ID, now uses first channel */
        E1432_GROUP_LIST_NODE *gn;
	gn = i1432_get_group_node(hw, ID);
	if (gn->chanlist == NULL)
	    /* But still error if there are no channels in the group */
	    return i1432_print_error(ERR1432_NO_CHANNEL);
	return i1432_get_module_from_chan(hw, gn->chanlist->chanID, mnp);
    }
    else
    {
        int     type, count;

        /* Get the channel type and count */
        type = (ID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
        count = (ID & E1432_CHAN_MASK) - 1;

        /* Do error checking */
        if (ID == 0 ||
	    type < 0 || type >= E1432_CHAN_TYPES ||
	    count < 0 || count >= i1432_chan_count[type])
	    return i1432_print_error(ERR1432_NO_CHANNEL);

        *mnp = i1432_chan_list[type][count].mn;
        return 0;
    }
}

SHORTSIZ16
i1432_get_chan_from_module(E1432_MODULE_LIST_NODE *mn)
{
    if (mn != NULL && mn->nchan > 0)
	return mn->chan_id;
    else
	return 0;
}

/*
 *********************************************************************
 Given a channel ID, return an index for use in channel arrays in
 modelistnodes.  No error checking is done on the parameter.
 *********************************************************************
 */
/*ARGSUSED*/
int
i1432_get_chan_index(E1432ID hw, SHORTSIZ16 ID)
{
    int     type, count;

    /* Get the channel type and count */
    type = (ID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
    count = (ID & E1432_CHAN_MASK) - 1;

    return i1432_chan_list[type][count].index;
}

/*
 *********************************************************************
 Given a channel ID, return the channel ID of a different channel
 within the same SCA.  No error checking is done on the parameter.  If
 there is no other channel in this SCA, then the function returns
 zero.
 *********************************************************************
 */
/*ARGSUSED*/
SHORTSIZ16
i1432_get_chan_other(E1432ID hw, SHORTSIZ16 ID)
{
    int     type, count;

    /* Get the channel type and count */
    type = (ID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
    count = (ID & E1432_CHAN_MASK) - 1;

    return i1432_chan_list[type][count].other_chan;
}

/*
 *********************************************************************
 Given a channel ID, return the SCA number of the channel within its
 module.  No error checking is done on the parameter.
 *********************************************************************
 */
/*ARGSUSED*/
int
i1432_get_chan_sca(E1432ID hw, SHORTSIZ16 ID)
{
    int     type, count;

    /* Get the channel type and count */
    type = (ID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
    count = (ID & E1432_CHAN_MASK) - 1;

    return i1432_chan_list[type][count].sca;
}

/*
 *********************************************************************
 Given a channel ID, return the SCA ID of the SCA containing this
 channel.  No error checking is done on the parameter.
 *********************************************************************
 */
/*ARGSUSED*/
int
i1432_get_chan_sca_id(E1432ID hw, SHORTSIZ16 ID)
{
    int     type, count;

    /* Get the channel type and count */
    type = (ID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
    count = (ID & E1432_CHAN_MASK) - 1;

    return i1432_chan_list[type][count].sca_id;
}

int
i1432_get_id_nchan(E1432ID hw, SHORTSIZ16 ID, int *nchan)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;

    if (ID < 0)
    {
	/* Group ID */
	*nchan = 0;
	gn = i1432_get_group_node(hw, ID);
	if (!gn)
	    return i1432_print_error(ERR1432_NO_GROUP);
	cn = gn->chanlist;
	while (cn != NULL)
	{
	    (*nchan)++;
	    cn = cn->next;
	}
    }
    else
	/* Channel ID */
	*nchan = 1;

    return 0;
}

/*
 *********************************************************************
 Returns the "system" module of a group ID.  The "system" module is
 that module on which the "pull sync line" action will be taken, in
 order to generate the measurement loop transitions.  If the ID is a
 channel ID, then return the module ID for that channel.
 *********************************************************************
 */
/*ARGSUSED*/
SHORTSIZ16
i1432_get_system_module(E1432ID hw, SHORTSIZ16 ID,
			E1432_MODULE_LIST_NODE **mnp)
{
    E1432_GROUP_LIST_NODE *gn;

    if (ID < 0)
    {
	/* Get group node */
	gn = i1432_get_group_node(hw, ID);
	if (!gn)
	    return i1432_print_error(ERR1432_NO_GROUP);

	/* Don't issue an error here if the system module is NULL.  It
	   may legitimately be NULL for a group that has no channels
	   in it. */

	*mnp = gn->system_mod;

	return 0;
    }
    else
	/* Channel ID, so use that module */
	return i1432_get_module_from_chan(hw, ID, mnp);
}

/*
 *********************************************************************
 Set the system channel in the e1432_group_list_node by looking at the
 group settings.

 Only one module needs to receive the sync command in a multi
 module system.  That module will be called the system module.

 Returns 0 if OK, error otherwise.
 *********************************************************************
 */
SHORTSIZ16
i1432_assign_system_module(E1432ID hw, SHORTSIZ16 groupID)
{
    E1432_CHAN_LIST_NODE *cn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error, last_chan;

    /* For channel ID, return immediately.  Don't error, to make it
       more convenient for e1432_init_measure to work with a channel
       ID. */
    if (groupID >= 0)
	return 0;

    gn = i1432_get_group_node(hw, groupID);
    if (!gn)
	return i1432_print_error(ERR1432_NO_GROUP);

    cn = gn->chanlist;
    last_chan = 0;
    while (cn != NULL)
    {
	last_chan = cn->chanID;	/* Save most recent channel number */
	cn = cn->next;
    }

    /* The system channel is the last channel in the chan list due to
       the read_raw_data() read ordering.  This is really left over
       from E1431, where this helped keep all modules in sync.  For
       E1432, it doesn't really matter except in multi-mainframe
       situations.

       In this case, having the system module be last in the list
       prevents the system module from moving ahead of modules in
       other mainframes.  However, this is not a complete solution
       because the system module can move ahead of the other modules
       anyway if there are no channels enabled in the mainframe that
       holds the system module.  Also, the system module can move
       ahead of the others in the ARM state as well, and possibly
       during measurement setup?  For multi-mainframe measurements,
       usually e1432_arm_measure_master_setup and related functions
       are needed to prevent problems.  Sometimes e1432_set_mmf_delay
       can be used instead to prevent problems.

       We used to set the system module differently when doing a
       local-bus measurement.  It used to be the module with the
       highest logical address in the group.  I never did know why
       this was done - it doesn't make any sense to me since logical
       addresses have no relation to slot number.  Anyway, this causes
       problems for multi-mainframe measurements, so I got rid of
       it. */

    if (last_chan > 0)
    {
	error = i1432_get_module_from_chan(hw, last_chan,
					   &gn->system_mod);
	if (error)
	    return error;
    }
    else
	/* A group with no channels has no system module */
	gn->system_mod = NULL;

    return 0;
}

/*
 *********************************************************************
 Check if valid ID.  Returns 0 if OK, else returns ERR1432_NO_CHANNEL
 or ERR1432_NO_GROUP.
 *********************************************************************
 */
SHORTSIZ16
i1432_checkID(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;

    if (ID < 0)
    {
	/* check group */
	gn = i1432_get_group_node(hw, ID);
	if (!gn)
	    return i1432_print_error(ERR1432_NO_GROUP);
    }
    else
    {
	/* check channel */
	error = i1432_get_module_from_chan(hw, ID, &mn);
	if (error)
	    return error;
    }
    return 0;
}



/*
 *********************************************************************
 Return selected info about a group
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_get_group_info(E1432ID hw, SHORTSIZ16 groupID, SHORTSIZ16 type,
		     SHORTSIZ16 *answer, SHORTSIZ16 answermax)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE  *cn;
    SHORTSIZ16 answersize=0;   /* current count of answers written */
    SHORTSIZ16 i=0;	       /* general temp variable */

    /* exit if bad input parameters */
    if(groupID>=0)
        return ERR1432_EXPECT_GROUP;

    if(!answer)
        return 0; /* you got what you asked for */

    if(answermax==0)
        return 0; /* you got what you asked for */

    /* get group node and channel node */
    gn=i1432_get_group_node(hw, groupID);
    if(!gn)
        return ERR1432_EXPECT_GROUP;

    cn=gn->chanlist;

    /* get only the selected info */
    switch (type)
    {
    case E1432_GROUP_INFO_NUM_MODULES:
	*answer=gn->modcount;
	break;
    case E1432_GROUP_INFO_LIST_MODULES:
	for(i=0;i<gn->modcount;i++)
	{
	    if(answersize<answermax)
	    {
		*answer=(gn->modlist[i])->la;
	        answer++;
	        answersize++;
	    }
	}
	break;
    case E1432_GROUP_INFO_NUM_CHANNELS:
	i=0;
	while(cn)
	{
	    i++;
	    cn=cn->next;
	}
	*answer=i;
	break;
    case E1432_GROUP_INFO_LIST_CHANNELS:
	while(cn)
	{
	    if(answersize<answermax)
	    {
		*answer=cn->chanID;
	        answer++;
		answersize++;
	    }
	    cn=cn->next;
	}
	break;
    case E1432_GROUP_INFO_NUM_INPUTS:
	i=0;
	while(cn)
	{
	    if(  ((cn->chanID)&E1432_CHAN_TYPE_MASK)
	       ==E1432_INPUT_CHAN(0) )
	    {
	        i++;
	    }
	    cn=cn->next;
	}
	*answer=i;
	break;
    case E1432_GROUP_INFO_LIST_INPUTS:
	while(cn)
	{
	    if(  ((cn->chanID)&E1432_CHAN_TYPE_MASK)
	       ==E1432_INPUT_CHAN(0) )
	    {
	        if(answersize<answermax)
	        {
		    *answer=cn->chanID;
	            answer++;
		    answersize++;
	        }
	    }
	    cn=cn->next;
	}
	break;
    case E1432_GROUP_INFO_NUM_SOURCES:
	i=0;
	while(cn)
	{
	    if(  ((cn->chanID)&E1432_CHAN_TYPE_MASK)
	       ==E1432_SOURCE_CHAN(0) )
	    {
	        i++;
	    }
	    cn=cn->next;
	}
	*answer=i;
	break;
    case E1432_GROUP_INFO_LIST_SOURCE:
	while(cn)
	{
	    if(  ((cn->chanID)&E1432_CHAN_TYPE_MASK)
	       ==E1432_INPUT_CHAN(0) )
	    {
	        if(answersize<answermax)
	        {
		    *answer=cn->chanID;
	            answer++;
		    answersize++;
	        }
	    }
	    cn=cn->next;
	}
	break;
    case E1432_GROUP_INFO_NUM_TACHS:
	i=0;
	while(cn)
	{
	    if(  ((cn->chanID)&E1432_CHAN_TYPE_MASK)
	       ==E1432_TACH_CHAN(0) )
	    {
	        i++;
	    }
	    cn=cn->next;
	}
	*answer=i;
	break;
    case E1432_GROUP_INFO_LIST_TACHS:
	while(cn)
	{
	    if(  ((cn->chanID)&E1432_CHAN_TYPE_MASK)
	       ==E1432_TACH_CHAN(0) )
	    {
	        if(answersize<answermax)
	        {
		    *answer=cn->chanID;
	            answer++;
		    answersize++;
	        }
	    }
	    cn=cn->next;
	}
	break;
    }
    return  0;
}

#ifdef	VXIPNP
LONGSIZ32
i1432_get_sicl_id(void)
{
    return i1432_mod_list->sicl_id;
}
#endif
